/**
 * \file: grl_bmpdec_jpeg.c
 *
 * version: $Id: grl_bmpdec_jpeg.c,v 1.9 2010/02/10 14:39:25 tkniep Exp $
 *
 * Implementation of the wrapper between the libjpeg and the
 * SVG Bitmap Decoder
 *
 * \component: SVG Bitmap Decoder (SVGBMPDEC)
 *
 * \author: T. Kniep (tkniep@de.adit-jv.com)
 *
 * \copyright: (c) 2009 ADIT Corporation
 *
 ***********************************************************************/

#include <string.h>
#include <svg_bitmap_decoder.h>
#include <grl_bitmap_decoder_util.h>
#include "grl_bmpdec_jpeg.h"
#include "drawjpeg.h"


#ifdef AMB_DEMO
    extern int ForcedOrientation;
#endif
/*******************************************************************************************
 *   Internal function prototypes
 *******************************************************************************************/

/*******************************************************************************************
 *  Local help functions
 */
const unsigned short	rotation_tbl[4][2] = {
	{1,	0},
	{6,	1},
	{3,	2},
	{8,	3},
};

unsigned char	jpeg_get_orientation( unsigned char image, unsigned short rotation_angle )
{
	unsigned char	loop_count;
	unsigned short	angle;
	unsigned char	ret_val;

	angle = 0;
	ret_val = 1;

	for (loop_count = 0; loop_count < 4; loop_count ++) {
		if (image == rotation_tbl[loop_count][0]) {
			angle = rotation_tbl[loop_count][1];
			break;
		}
	}

	angle = (angle + rotation_angle) % 4;

	for (loop_count = 0; loop_count < 4; loop_count ++) {
		if (angle == rotation_tbl[loop_count][1]) {
			ret_val = (unsigned char)rotation_tbl[loop_count][0];
			break;
		}
	}

	return(ret_val);
}

/*******************************************************************************************
 *   Function implementations
 *******************************************************************************************/

SVGError GRL_BMPDEC_draw_image_JPEG ( GRL_BMPDEC_dec_info *p_decode_info )
{
    const     PIX_RGB         pixel_info_ARGB8888 = {4, {16, 23}, {8, 15}, {0, 7}};

    SVGError        ret_err         = SVG_NO_ERROR;
    DRAW_AREA_INFO  draw_area_info  = {0,{0,{0,0},{0,0},{0,0}},{0,0},0};                  /* draw area information                */
    DRAW_IMG_INFO   draw_img_info   = {0,{0,0},{0,0},0,0,0,0,0,0,{{0,0,0},0},0};    /* draw image information               */
    JPEG_IMG_INFO   jpeg_img_info   = {0,0,0,{0,0,0},{0}};                  /* Image info read from JPEG            */
    SVGImage        *p_image        = NULL;                 /* Pointer to Image str                 */
    SVGUint8        *p_dst_mem      = NULL;                 /* Pointer to destination memory        */
    SVGUint8*       p_data          = NULL;                 /* Pointer to data                      */
    SVGUint32       dst_mwidth      = 0;                    /* Destination memory width             */
    SVGInt32        dst_xpos        = 0;
    SVGInt32        dst_ypos        = 0;
    SVGUint32       dst_area_width  = 0;                    /* destination x size of target surface */
    SVGUint32       dst_draw_width  = 0;                    /* destination x size of image          */
    SVGUint32       dst_draw_height = 0;                    /* destination y size of image          */
    SVGUint16       src_width       = 0;                    /* Width of source image                */
    SVGUint16       src_height      = 0;                    /* Height of source image               */
    SVGUint32       ret_jpeg_dec    = 0;                    /* function return value                */
    SVGInt32        stop_flag       = 0;
    SVGInt32        pix             = 0;
    SVGBmpDecRect   src_clip        = {0,0,0,0};                  /* Source clip rect                     */
    SVGInt32		clip_x			= 0;
    SVGInt32		clip_y			= 0;
    SVGInt32		clip_width		= 0;
    SVGInt32		clip_height 	= 0;
    SVGBoolean      is_clipped      = SVG_FALSE;            /* Image has to be clipped              */
    SVGInt32        reg_jpeg_dec    = 0;
    SVGInt32        zoom_x          = 0;
    SVGInt32        zoom_y          = 0;
    SVGInt32        zoom            = 0;
    SVGUint16       tmp_swap        = 0;
    SVGBoolean      is_clip_prov    = 1; 					/*is clip rectengle provided by user(default true)*/

    int				exif_orientation;
    SVG_BMP_U("Enter into GRL_BMPDEC_DRAW_IMAGE_JPEG");
        
    if ( p_decode_info == NULL )
    {
        ret_err = SVG_POINTER_NULL;
        SVG_BMP_E("SVG_POINTER_NULL IN GRL_BMPDEC_DRAW_IMAGE_JPEG");
    }
    else if ( 0 == p_decode_info->image.data_size )
    {
        /* The JPG decoder may crash if data_size == 0 */
        ret_err = SVG_INVALID_VALUE;
        SVG_BMP_E("SVG_INVALID_VALUE IN GRL_BMPDEC_DRAW_IMAGE_JPEG");
    }

    if ( SVG_NO_ERROR == ret_err )
    {
/* PRQA: Lint Message 613: Pointer p_decode_info was checked but Lint does not understand the condition with ret_err */
/*lint -save -e613 */
/* PRQA: QAC Message 505: Pointer p_decode_info was checked but QAC does not understand the condition with ret_err */
        p_image = &(p_decode_info->image);
/*lint -restore */
        /* Handle destination-specific settings */

        /* Handle destination-specific settings */
        switch ( p_image->decode_mode )
        {
            case SVG_BMPDEC_MODE_MEMORY:
            {
                p_dst_mem       = p_image->dest.dest_mem.dest_mem;
                dst_area_width  = p_image->width;    /* Assume width to be the width of the image */
                dst_mwidth      = dst_area_width * GRL_BMPDEC_DEPTH_ARGB8888;
                dst_xpos        = 0;                 /* Position is always at beginning of mem block because */
                dst_ypos        = 0;                 /* otherwise the size might not be sufficient */
                p_data          = (U8*)((void*)p_image->datap);
                clip_x = p_image->clip.x;
                clip_y = p_image->clip.y;
                clip_width = p_image->clip.width;
                clip_height = p_image->clip.height;
                break;
            }
            default:
            {
                ret_err = SVG_INVALID_OPERATION;
                SVG_BMP_E("SVG_INVALID_OPERATION IN GRL_BMPDEC_DRAW_IMAGE_JPEG");
            }
        }
    }
    if ( (SVG_NO_ERROR == ret_err) &&
         (p_data == NULL) )
    {
        ret_err = SVG_INVALID_VALUE;
        SVG_BMP_E("SVG_INVALID_VALUE IN GRL_BMPDEC_DRAW_IMAGE_JPEG");
    }
    if ( (SVG_NO_ERROR == ret_err) && (p_image != NULL))
    {
        dst_draw_width  = p_image->width;
        dst_draw_height = p_image->height;
        
        draw_img_info.type           = IMG_TYP_JPEG;
        draw_img_info.datap          = (U8 *)((void *)p_image->datap);
        draw_img_info.datasize       = p_image->data_size;
        if ( 0 < p_image->data_size )
        {
            reg_jpeg_dec = (SVGInt32)_GetImageJPEG( &draw_img_info, &jpeg_img_info);
            if ( reg_jpeg_dec != 0 )
            {
                ret_err = SVG_BMPDEC_DEC_ERROR;
                SVG_BMP_E("SVG_BMPDEC_DEC_ERROR IN GRL_BMPDEC_DRAW_IMAGE_JPEG");
            }
            else
            {
				src_width  = (SVGUint16)jpeg_img_info.width;
				src_height = (SVGUint16)jpeg_img_info.height;
            }
        }
        else
        {
            ret_err = SVG_INVALID_VALUE;
            SVG_BMP_E("SVG_INVALID_VALUE IN GRL_BMPDEC_DRAW_IMAGE_JPEG");
        }
    }

    if ( SVG_NO_ERROR == ret_err )
    {
        /* If no src_clip is set, use the size of the source image */
        if ( (clip_x == 0) && (clip_y == 0) && (clip_width == 0) && (clip_height == 0) )
    	{
            clip_width = src_width;
            clip_height = src_height;
            is_clip_prov = 0;
        }
        /* orientation */

        exif_orientation = jpeg_img_info.orientation;

#ifdef AMB_DEMO
    exif_orientation=ForcedOrientation;
#endif
/*PRQA: Lint Message 613: Pointer p_decode_info was checked but Lint does not understand the condition with ret_err */
/*lint -save -e613 */
        exif_orientation = jpeg_get_orientation(exif_orientation, p_decode_info->image.attribute.jpeg.rotation);
/*lint -restore */

        switch ( exif_orientation )
		{
			case 6:
			{
				tmp_swap    = clip_width;
				clip_width  = clip_height;
				clip_height = tmp_swap;
				if( is_clip_prov )
				{
					tmp_swap    = clip_y;
					clip_y      = (src_height - clip_height) - tmp_swap;
					if(clip_y < 0)
					{
						clip_height += clip_y;
						clip_y       = (src_height - clip_height) - tmp_swap;
					}
					clip_x      = tmp_swap;
				}
				break;
			}
			case 3:
			{
				clip_x      = (src_width  - clip_width)  - clip_x;
				clip_y      = (src_height - clip_height) - clip_y;
				break;
			}
			case 8:
			{
				tmp_swap    = clip_width;
				clip_width  = clip_height;
				clip_height = tmp_swap;
				if( is_clip_prov )
				{
					tmp_swap    = clip_x;
					clip_x      = (src_width - clip_width) - clip_y;
					if(clip_x < 0)
					{
						clip_width += clip_x;
						clip_x      = (src_width - clip_width) - clip_y ;
					}
					clip_y      = tmp_swap;
				}
				break;
			}
			default:
			{
				break;
			}
		}
        /* Calculate scale factor between src clip size and destination size */
        if( (clip_width != 0) && (clip_height != 0) )
        {
            zoom_x = (S32)(((U32)(dst_draw_width)  * GRL_BMPDEC_ZOOM_ORG) / (U32)clip_width);
            zoom_y = (S32)(((U32)(dst_draw_height) * GRL_BMPDEC_ZOOM_ORG) / (U32)clip_height);
        }

        if ( zoom_x < zoom_y )
        {
            zoom = zoom_x;
        }
        else
        {
            zoom = zoom_y;
        }
        if ( zoom < 1 )
        {
            zoom = 1;
        }

        src_clip.x      = clip_x;
        src_clip.y      = clip_y;
        src_clip.width  = clip_width;
        src_clip.height = clip_height;
        if( exif_orientation == 6 || exif_orientation == 8)
        {
			ret_err = GRL_BMPDEC_clipping_setup( (SVGInt32)src_height,(SVGInt32)src_width,
														 &dst_xpos, &dst_ypos,
														 &src_clip,
														 src_width, src_height,
														 &is_clipped );
        }
        else
        {
			ret_err = GRL_BMPDEC_clipping_setup( (SVGInt32)src_width,(SVGInt32)src_height,
														 &dst_xpos, &dst_ypos,
														 &src_clip,
														 src_width, src_height,
														 &is_clipped );

        }

        draw_img_info.image_effect.rotation = exif_orientation;
    }
    if ( SVG_NO_ERROR == ret_err )
    {
        /* Set draw area information (only ARGB8888 is supported as destination) */
        draw_area_info.mwidth           = (SVGInt16)dst_mwidth;
        draw_area_info.vram_adrs        = p_dst_mem;
        draw_area_info.size.x           = (SVGInt16)dst_draw_width;
        draw_area_info.size.y           = (SVGInt16)dst_draw_height;
        draw_area_info.rgb 				= pixel_info_ARGB8888;
        pix                				= 4;//SVG_RES_SURFACE_DEPTH32;

        /* Compute start address in destination memory */
        draw_area_info.vram_adrs        = &draw_area_info.vram_adrs[(dst_xpos * pix)];
        draw_area_info.vram_adrs        = &draw_area_info.vram_adrs[(dst_ypos * draw_area_info.mwidth)];

        /* Set draw image information */
/*PRQA: Lint Message 613: Pointer p_decode_info was checked but Lint does not understand the condition with ret_err */
/*lint -save -e613 */
        if ( p_decode_info->image.type == SVG_BMPDEC_JPEG_ENCODING )
        {
/*lint -restore */
            draw_img_info.type              = IMG_TYP_JPEG;
        }
        else
        {
            draw_img_info.type              = IMG_TYP_THNMB;
        }

        draw_img_info.org.x         = src_clip.x;
        draw_img_info.org.y         = src_clip.y;
        draw_img_info.size.x        = (SVGInt16)src_clip.width;
        draw_img_info.size.y        = (SVGInt16)src_clip.height;
        draw_img_info.zx            = zoom;
        draw_img_info.zy            = zoom;
/*PRQA: Lint Message 613: Pointer p_decode_info was checked but Lint does not understand the condition with ret_err */
/*lint -save -e613 */
        draw_img_info.datasize      = p_decode_info->image.data_size;
        draw_img_info.datap         = p_data;

        /* Handle tone effects */
        if ( p_decode_info->image.attribute.jpeg.effect_enable == SVG_TRUE )
        {
            draw_img_info.image_effect.tone_effect.effect_enable = TONE_EFFECT_ENABLE;
            draw_img_info.image_effect.tone_effect.U             = p_decode_info->image.attribute.jpeg.tone_u;
            draw_img_info.image_effect.tone_effect.V             = p_decode_info->image.attribute.jpeg.tone_v;
/*lint -restore */

        }
        else
        {
            draw_img_info.image_effect.tone_effect.effect_enable = TONE_EFFECT_DISABLE;
            draw_img_info.image_effect.tone_effect.U             = 0;
            draw_img_info.image_effect.tone_effect.V             = 0;
        }

        if(p_decode_info){
        draw_img_info.revert = p_decode_info->revert;
        }
        /* Draw image */
        ret_jpeg_dec = _PutImageJPEG( &draw_area_info,
                                      &draw_img_info,
                                      &stop_flag);
        if ( ret_jpeg_dec != JPEG_SUCCESS )
        {
            ret_err = SVG_BMPDEC_DEC_ERROR;
            SVG_BMP_E("SVG_BMPDEC_DEC_ERROR IN GRL_BMPDEC_DRAW_IMAGE_JPEG");
        }
    }

    SVG_BMP_U("EXIT FROM GRL_BMPDEC_DRAW_IMAGE_JPEG");

    return  ret_err;
}

SVGError GRL_BMPDEC_get_image_info_JPEG ( const SVGContextBmpDec  *ctx,
                                          const SVGImage          *p_image,
                                          SVGImageInfo            *image_info )
{
    SVGError            ret_err         = SVG_NO_ERROR;
    SVGInt32            reg_jpeg_dec    = 0;
    DRAW_IMG_INFO       draw_img        = {0,{0,0},{0,0},0,0,0,0,0,0,{{0,0,0},0}, 0};
    JPEG_IMG_INFO       jpeg_img_info   = {0,0,0,{0,0,0},{0}};

    SVG_BMP_U("ENTER INTO GRL_BMPDEC_GET_IMAGE_INFO_JPEG");

    if ( (ctx == NULL) || (p_image == NULL) || (image_info == NULL) )
    {
        ret_err = SVG_POINTER_NULL;
        SVG_BMP_E("SVG_POINTER_NULL IN GRL_BMPDEC_GET_IMAGE_INFO_JPEG");
    }
    else
    {
        if ( p_image->datap == NULL )
        {
            ret_err = SVG_POINTER_NULL;
            SVG_BMP_E("SVG_POINTER_NULL IN GRL_BMPDEC_GET_IMAGE_INFO_JPEG");
        }
        else
        {
            draw_img.type           = IMG_TYP_JPEG;
            draw_img.datap          = (U8 *)((void *)p_image->datap);
            draw_img.datasize       = p_image->data_size;
            if ( 0 < p_image->data_size )
            {
                reg_jpeg_dec = (SVGInt32)_GetImageJPEG( &draw_img, &jpeg_img_info);
                if ( reg_jpeg_dec != 0 )
                {
                    ret_err = SVG_BMPDEC_DEC_ERROR;
                    SVG_BMP_E("SVG_BMPDEC_DEC_ERROR IN GRL_BMPDEC_GET_IMAGE_INFO_JPEG");
                }
                else
                {
                    memset( image_info, 0, sizeof(SVGImageInfo) );
                    image_info->type                        = SVG_BMPDEC_JPEG_ENCODING;
                    image_info->width                       = (SVGUint16)jpeg_img_info.width;
                    image_info->height                      = (SVGUint16)jpeg_img_info.height;
                    image_info->bit_depth                   = GRL_BMPDEC_JPEG_DEPTH;            /* constant depth */
                    image_info->format.jpeg.orientation     = jpeg_img_info.orientation;
                    memcpy( image_info->format.jpeg.datetime, jpeg_img_info.datetime, GRL_BMPDEC_JPEG_DTIME_LEN );
                }
            }
            else
            {
                /* The JPG decoder may crash if data_size == 0 */
                ret_err = SVG_INVALID_VALUE;
                SVG_BMP_E("SVG_INVALID_VALUE IN GRL_BMPDEC_GET_IMAGE_INFO_JPEG");
            }                   
        }
    }       

    SVG_BMP_U("EXIT FROM GRL_BMPDEC_GET_IMAGE_INFO_JPEG");

    return  ret_err;
}
